/*
 * ctdb_tutorial3.c
 *
 * Public domain c-treeDB C example
 *
 * FairCom Corporation, 6300 West Sugar Creek Drive, Columbia, MO 65203 USA
 *
 * The goal of this tutorial is to introduce the most basic CTDB API
 * to accomplish creating and manipulating a table through the ctreeServer
 *
 * Functionally, this application will perform the following:
 *  1.  Logon onto a session
 *  2.  Add 1 table with some fields
 *  3.  Populate the table with a few records
 *  4.  Display the contents of the table
 *  5.  Update one record under locking control
 */

#ifdef _WIN32_WCE
#undef UNICODE
#undef _UNICODE
#define main my_main
#endif

/* Preprocessor definitions and includes */

#include "ctdbsdk.h" /* c-tree headers */

#define LOCK_SUPPORT

#ifndef ctCLIENT
#ifdef NOTFORCE
#undef LOCK_SUPPORT
#endif
#endif

#define END_OF_FILE INOT_ERR   /* INOT_ERR is ctree's 101 error. See cterrc.h */

/////////// MY ADDED //////////
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <ifaddrs.h>

#define MSG_SIZE 40			// message size

int data_tmp,order_qty,order_ticket;
int status_flag;

void error(const char *msg)
{
    perror(msg);
    exit(0);
}
/////////////////////////////////



/* Global declarations */

CTHANDLE hSession;
CTHANDLE hDatabase;
CTHANDLE hTable;
CTHANDLE hRecord;


/* Function declarations */

#ifdef PROTOTYPE
VOID Initialize(VOID), Define(VOID), Manage(VOID), Done(VOID);
VOID Add_Ticket_Records(VOID), Display_Records(VOID);
VOID Update_Ticket_Record(VOID), Create_Ticket_Table(VOID);
VOID Delete_Records(CTHANDLE), Check_Table_Mode(CTHANDLE);
VOID Handle_Error(CTSTRING);
#else
VOID Initialize(), Define(), Manage(), Done();
VOID Add_Ticket_Records(), Display_Records();
VOID Delete_Records(), Check_Table_Mode();
VOID Update_Ticket_Record(), Create_Ticket_Table();
VOID Handle_Error();
#endif


/*
 * main()
 *
 * The main() function implements the concept of "init, define, manage
 * and you're done..."
 */

#ifdef PROTOTYPE
NINT main (NINT argc, pTEXT argv[])
#else
NINT main (argc, argv)
NINT argc;
TEXT argv[];
#endif
{

		///// declare variables for socket /////
		int sock, length, n;
		int boolval = 1;			// for a socket option
		socklen_t fromlen;
		struct sockaddr_in server;
		struct sockaddr_in addr;
		char buffer[MSG_SIZE];	// to store received messages or messages to be sent.
		char s_buffer[MSG_SIZE];
		char station_s[5],id_s[5],qty_s[5], ticket_s[5], transcation[5];
		char *token_s;

		order_ticket = 1;
		/////////////////////////////////////////


		/////////Initialize FairCom Database////////

		 Initialize();

		 Define();

	     Manage();

	     /////////Initialize Socket////////

		 sock = socket(AF_INET, SOCK_DGRAM, 0); // Creates socket. Connectionless.
		 if (sock < 0)
		   error("Opening socket");

		 length = sizeof(server);			// length of structure
		 bzero(&server,length);			// sets all values to zero. memset() could be used
		 server.sin_family = AF_INET;		// symbol constant for Internet domain
		 server.sin_addr.s_addr = INADDR_ANY;		// IP address of the machine on which
													// the server is running
		 server.sin_port = htons(3000);	// using port 3000

		 // binds the socket to the address of the host and the port number
		 if (bind(sock, (struct sockaddr *)&server, length) < 0)
			 error("binding");

		 // change socket permissions to allow broadcast
		 if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &boolval, sizeof(boolval)) < 0)
		 {
		   	printf("error setting socket options\n");
		   	exit(-1);
		 }

		 fromlen = sizeof(struct sockaddr_in);	// size of structure


		 /////////Receive and handle the orders////////
	   while (1)
	   {
		   // to "clean up" the buffer. The messages aren't always the same length...
		   bzero(buffer,MSG_SIZE);

		   // receive from a client
		   n = recvfrom(sock, buffer, MSG_SIZE, 0, (struct sockaddr *)&addr, &fromlen);
		   if (n < 0)
		   	   error("recvfrom");
	       printf("Received: %s", buffer);

	       //usleep(10000);

	       token_s = strtok(buffer, " ");
	       token_s = strtok(NULL, " ");
	       strcpy(station_s, token_s); //Station ID
	       token_s = strtok(NULL, " ");
	       token_s = strtok(NULL, " ");
	       strcpy(id_s, token_s); //Thread ID from a clien
	       token_s = strtok(NULL, " ");
	       token_s = strtok(NULL, " ");
	       strcpy(ticket_s, token_s); //Ticket requested
	       token_s = strtok(NULL, " ");
	       token_s = strtok(NULL, " ");
	       strcpy(qty_s, token_s); //Quantity requested
	       token_s = strtok(NULL, " ");
	       token_s = strtok(NULL, " ");
	       strcpy(transcation, token_s); //Transaction ID
	       //printf("Received order from station%s id%s ticket:%s qty:%s\n", station_s, id_s,ticket_s, qty_s);

	       order_qty = atoi(qty_s);
	       order_ticket = atoi(ticket_s);



	        /* display and check total ticket left */
	        Display_Records();

		    /* update ticket number */
		    Update_Ticket_Record();

		    /* Show total left*/
		    Display_Records();

		    if(status_flag == 1) //this flag is from Update_ticket_record()
		    	//sprintf(s_buffer,"%s order(s) of ticket %s done\n",qty_s,ticket_s);
		    	sprintf(s_buffer,"Approved trns:%s\n", transcation);
		    else
		    	sprintf(s_buffer,"Ticket %s SOLD OUT!\n",ticket_s);
	       n = sendto(sock, s_buffer, 40, 0,
	             (struct sockaddr *)&addr, fromlen); //send result to cilent
	       if (n  < 0)
	         error("sendto");
	   }

   return(0);
}


/*
 * Initialize()
 *
 * Perform the minimum requirement of logging onto the c-tree Server
 */

#ifdef PROTOTYPE
VOID Initialize(VOID)
#else
VOID Initialize()
#endif
{
   CTDBRET  retval;

   printf("INIT\n");

   if ((retval = ctdbStartDatabaseEngine())) 				/* This function is required when you are using the Server DLL model to start the underlying Server. */
	Handle_Error("Initialize(): ctdbStartDatabaseEngine()"); 	/* It does nothing in all other c-tree models */

   /* allocate session handle */
   if ((hSession = ctdbAllocSession(CTSESSION_CTREE)) == NULL)
      Handle_Error("Initialize(): ctdbAllocSession()");

   hDatabase = hSession; /* database not used in this tutorial */

   /* connect to server */
   printf("\tLogon to server...\n");
   if (ctdbLogon(hSession, "FAIRCOMS", "ADMIN", "ADMIN"))
      Handle_Error("Initialize(): ctdbLogon()");
}


/*
 * Define()
 *
 * Open the table, if they exist. Otherwise create and open the table
 */

#ifdef PROTOTYPE
VOID Define(VOID)
#else
VOID Define()
#endif
{
   printf("DEFINE\n");

   Create_Ticket_Table();
}


/*
 * Manage()
 *
 * This function performs record adds and updates using locking
 */

#ifdef PROTOTYPE
VOID Manage(VOID)
#else
VOID Manage()
#endif
{
   printf("MANAGE\n");

   /* allocate a record handle */
   if ((hRecord = ctdbAllocRecord(hTable)) == NULL)
      Handle_Error("Manage(): ctdbAllocRecord()");

   /* delete any existing records */
   Delete_Records(hRecord);

   /* populate the table with data */
   Add_Ticket_Records();

   /* display contents of table */
   Display_Records();
}


/*
 * Done()
 *
 * This function handles the housekeeping of closing tables and
 * freeing of associated memory
 */

#ifdef PROTOTYPE
VOID Done(VOID)
#else
VOID Done()
#endif
{
   printf("DONE\n");

   /* close table */
   printf("\tClose table\n");
   if (ctdbCloseTable(hTable))
      Handle_Error("Done(): ctdbCloseTable()");

   /* logout */
   printf("\tLogout...\n");
   if (ctdbLogout(hSession))
      Handle_Error("Done(): ctdbLogout()");

   /* free handles */
   ctdbFreeRecord(hRecord);
   ctdbFreeTable(hTable);
   ctdbFreeSession(hSession);

   /* If you are linked to the Server DLL, then we should stop our Server at the end of the program.   */
   /* It does nothing in all other c-tree models */
   ctdbStopDatabaseEngine();
}


/*
 * Create_CustomerMaster_Table()
 *
 * Open table CustomerMaster, if it exists. Otherwise create it
 * along with its indices and open it
 */

#ifdef PROTOTYPE
VOID Create_Ticket_Table(VOID)
#else
VOID Create_Ticket_Table()
#endif
{
   CTHANDLE pField1, pField2;
   CTHANDLE pIndex;
   CTHANDLE pIseg;

   /* define table CustomerMaster */
   printf("\tcreate Ticket table\n");

   /* allocate a table handle */
   if ((hTable = ctdbAllocTable(hDatabase)) == NULL)
      Handle_Error("Create_Ticket_Table(): ctdbAllocTable()");

   /* open table */
   if (ctdbOpenTable(hTable, "tblticket", CTOPEN_NORMAL))
   {
      /* define table fields */
      pField1 = ctdbAddField(hTable, "ticket_id", CT_STRING, 4);
      pField2 = ctdbAddField(hTable, "ticket_qty", CT_STRING, 10);

      if (!pField1 || !pField2 )
         Handle_Error("Create_Ticket_Table(): ctdbAddField()");

      /* define index */
      pIndex = ctdbAddIndex(hTable, "tblticket_idx", CTINDEX_FIXED, NO, NO);
      pIseg = ctdbAddSegment(pIndex, pField1, CTSEG_SCHSEG);
      if (!pIndex || !pIseg)
         Handle_Error("Create_Ticket_Table(): ctdbAddIndex()|ctdbAddSegment()");

      /* create table */
      if (ctdbCreateTable(hTable, "tblticket", CTCREATE_NORMAL))
         Handle_Error("Create_Ticket_Table(): ctdbCreateTable()");

      /* open table */
      if (ctdbOpenTable(hTable, "tblticket", CTOPEN_NORMAL))
         Handle_Error("Create_Ticket_Table(): ctdbOpenTable()");
   }
   else
   {
      Check_Table_Mode(hTable);

      /* confirm the index exists, if not then add the index
       *
       * this scenario arises out of the fact that this table was created in tutorial 1
       * without indexes. The index is now created by the call to ctdbAlterTable
       */

      if (ctdbGetIndexByName(hTable, "tblticket_idx") == NULL)
      {
         pField1 = ctdbGetFieldByName(hTable, "tblticket");
         pIndex = ctdbAddIndex(hTable, "tblticket_idx", CTINDEX_FIXED, NO, NO);
         pIseg = ctdbAddSegment(pIndex, pField1, CTSEG_SCHSEG);
         if (!pIndex || !pIseg)
            Handle_Error("Create_Ticket_Table(): ctdbAddIndex()|ctdbAddSegment()");

         if (ctdbAlterTable(hTable, CTDB_ALTER_NORMAL) != CTDBRET_OK)
            Handle_Error("Create_Ticket_Table(): ctdbAlterTable()");
      }
   }
}


/*
 * Check_Table_Mode()
 *
 * Check if existing table has transaction processing flag enabled.
 * If a table is under transaction processing control, modify the
 * table mode to disable transaction processing
 */

#ifdef PROTOTYPE
VOID Check_Table_Mode(CTHANDLE hTable)
#else
VOID Check_Table_Mode(hTable)
CTHANDLE hTable;
#endif
{
   CTCREATE_MODE mode;

   /* get table create mode */
   mode = ctdbGetTableCreateMode(hTable);

   /* check if table is under transaction processing control */
   if ((mode & CTCREATE_TRNLOG))
   {
      /* change file mode to disable transaction processing */
      mode ^= CTCREATE_TRNLOG;
      if (ctdbUpdateCreateMode(hTable, mode) != CTDBRET_OK)
         Handle_Error("Check_Table_Mode(); ctdbUpdateCreateMode");
   }
}


/*
 * Delete_Records()
 *
 * This function deletes all the records in the table
 */

#ifdef PROTOTYPE
VOID Delete_Records(CTHANDLE hRecord)
#else
VOID Delete_Records(hRecord)
CTHANDLE hRecord;
#endif
{
   CTDBRET  retval;
   CTBOOL   empty;

   printf("\tDelete records...\n");

   /* enable session-wide lock flag */
   if (ctdbLock(hSession, CTLOCK_WRITE_BLOCK))
      Handle_Error("Delete_Records(): ctdbLock()");

   empty = NO;
   retval = ctdbFirstRecord(hRecord);
   if (retval != CTDBRET_OK)
   {
      if (retval == END_OF_FILE)
         empty = YES;
      else
         Handle_Error("Delete_Records(): ctdbFirstRecord()");
   }

   while (empty == NO) /* while table is not empty */
   {
      /* delete record */
      if (ctdbDeleteRecord(hRecord))
         Handle_Error("Delete_Records(): ctdbDeleteRecord()");

      /* read next record */
      retval = ctdbNextRecord(hRecord);
      if (retval != CTDBRET_OK)
      {
         if (retval == END_OF_FILE)
            empty = YES;
         else
            Handle_Error("Delete_Records(): ctdbNextRecord()");
      }
   }
   if (ctdbUnlock(hSession))
      Handle_Error("Delete_Records(): ctdbUnlock()");
}


/*
 * Add_CustomerMaster_Records()
 *
 * This function adds records to table CustomerMaster from an
 * array of strings
 */

typedef struct {
   CTSTRING id, qty;
} CUSTOMER_DATA;


///////////// Initialize total ticket number ////////////////
CUSTOMER_DATA data[] = {
   "1", "500",
   "2", "500"
};
/////////////////////////////////////////////////////////////

#ifdef PROTOTYPE
VOID Add_Ticket_Records(VOID)
#else
VOID Add_Ticket_Records()
#endif
{
   CTDBRET  retval;
   CTSIGNED i;
   CTSIGNED nRecords = sizeof(data) / sizeof(CUSTOMER_DATA);

   printf("\tAdd records...\n");

   /* add data to table */
   for (i = 0; i < nRecords; i++)
   {
      /* clear record buffer */
      ctdbClearRecord(hRecord);

      retval = 0;
      /* populate record buffer with data */
      retval |= ctdbSetFieldAsString(hRecord, 0, data[i].id);
      retval |= ctdbSetFieldAsString(hRecord, 1, data[i].qty);

      if (retval)
         Handle_Error("Add_Ticket_Records(): ctdbSetFieldAsString()");

      /* add record */
      if (ctdbWriteRecord(hRecord))
         Handle_Error("Add_Ticket_Records(): ctdbWriteRecord()");
   }
}


/*
 * Display_Records()
 *
 * This function displays the contents of a table. ctdbFirstRecord() and
 * ctdbNextRecord() fetch the record. Then each field is parsed and displayed
 */

#ifdef PROTOTYPE
VOID Display_Records(VOID)
#else
VOID Display_Records()
#endif
{
   CTDBRET  retval;
   TEXT     ticketid[4+1];
   TEXT     ticketqty[47+1];
   int x;

 //  printf("\tDisplay records...");

   /* read first record */
   retval = ctdbFirstRecord(hRecord);
   if (retval == END_OF_FILE)
      return;

   x = 1;
   while (retval == CTDBRET_OK)
   {
      retval = 0;
      retval |= ctdbGetFieldAsString(hRecord, 0, ticketid, sizeof(ticketid));
      retval |= ctdbGetFieldAsString(hRecord, 1, ticketqty, sizeof(ticketqty));
      if (retval)
         Handle_Error("Display_Records(): ctdbGetFieldAsString()");

 //     printf("\n\t\t%-8s%10s\n",ticketid, ticketqty);
      printf("Ticket %s remaining: %s\n",ticketid, ticketqty);
      if(order_ticket == x)
    	  data_tmp = atoi(ticketqty);

      /* read next record */
      retval = ctdbNextRecord(hRecord);
      if (retval == END_OF_FILE)
         break;   /* reached end of file */

      if (retval != CTDBRET_OK)
         Handle_Error("Display_Records(): ctdbNextRecord()");

      x++;
   }
}


/*
 * Update_CustomerMaster_Records()
 *
 * Update one record under locking control to demonstrate the effects
 * of locking
 */

#ifdef PROTOTYPE
VOID Update_Ticket_Record(VOID)
#else
VOID Update_Ticket_Record()
#endif
{

	char buffer[10],inx_finder[5];
 //  printf("\tUpdate record...\n");

   /* enable session-wide lock flag */
   if (ctdbLock(hSession, CTLOCK_WRITE_BLOCK))
      Handle_Error("Update_Ticket_Record(): ctdbLock()");

   /* find record by customer number */
   if (ctdbClearRecord(hRecord))
      Handle_Error("Update_Ticket_Record(): ctdbClearRecord()");

   sprintf(inx_finder,"%d",order_ticket);
  // printf("index:%s ",inx_finder);

   //if (ctdbSetFieldAsString(hRecord, 0, "2"))
   if (ctdbSetFieldAsString(hRecord, 0, inx_finder))
      Handle_Error("Update_Ticket_Record(): ctdbSetFieldAsString()");
   if (ctdbFindRecord(hRecord, CTFIND_EQ))
      Handle_Error("Update_Ticket_Record(): ctdbFindRecord()");

  // printf("%d - %d", data_tmp,order_qty);

   if(data_tmp-order_qty >= 0){
	   data_tmp = data_tmp-order_qty;
	   status_flag = 1;
   }
   else {
	   data_tmp = data_tmp;
	   status_flag = 0;
   }

   printf("Trying to sell %d order(s) of ticket %d.......\n", order_qty,order_ticket);

   sprintf(buffer,"%d",data_tmp);



   if (ctdbSetFieldAsString(hRecord, 1, buffer))
      Handle_Error("Update_Ticket_Record(): ctdbSetFieldAsString()");
   /* rewrite record */
   if (ctdbWriteRecord(hRecord))
      Handle_Error("Update_Ticket_Record(): ctdbWriteRecord()");
   else
   {
      //printf("\tPress <ENTER> key to unlock\n");
      //getchar();
   }

   if (ctdbUnlock(hSession))
      Handle_Error("Update_Ticket_Record(): ctdbUnlock()");
}

/*
 * Handle_Error()
 *
 * This function is a common bailout routine. It displays an error message
 * allowing the user to acknowledge before terminating the application
 */

#ifdef PROTOTYPE
VOID Handle_Error(CTSTRING errmsg)
#else
VOID Handle_Error(errmsg)
CTSTRING errmsg;
#endif
{
   printf("\nERROR: [%d] - %s \n", ctdbGetError(hSession), errmsg);
   printf("*** Execution aborted *** \nPress <ENTER> key to exit...");

   ctdbLogout(hSession);

   ctdbFreeRecord(hRecord);
   ctdbFreeTable(hTable);
   ctdbFreeSession(hSession);

   getchar();

   exit(1);
}

/* end of ctdb_tutorial3.c */
